#
# This is an example VCL file for Varnish.
#
# It does not do anything by default, delegating control to the
# builtin VCL. The builtin VCL is called when there is no explicit
# return statement.
#
# See the VCL chapters in the Users Guide at https://www.varnish-cache.org/docs/
# and http://varnish-cache.org/trac/wiki/VCLExamples for more examples.

# Marker to tell the VCL compiler that this VCL has been adapted to the
# new 4.0 format.
vcl 4.0;

import std;
import cookie;
import header;

backend default {
    .host = "127.0.0.1";
    .port = "PORT";
    .probe = {
        .url = "/home";
        .timeout = 5s;
        .interval = 5s;
        .window = 5;
        .threshold = 3;
    }
}

sub vcl_recv {
 	
 	# Happens before we check if we have this in cache already.
 	#
 	# Typically you clean up the request here, removing cookies you don't need,
 	# rewriting the request, etc.

	# Strip out query parameters such as 't' that do not affect the page content
	# set req.url = regsuball(req.url, "([\?|\&])+(t)=[^&\s]+", "\1");

 	if (req.url ~ "\.(png|gif|jpg|swf|css|js)$" || req.url ~ "^/documents/") {
  		set req.url = regsuball(req.url, "([\?|\&])+(t)=[^&\s]+", "\1");
		#  std.syslog(8+1, "MHV.REQ.URL_VCL_RECV: " + req.url);
  		return(hash);
 	}
}

sub vcl_hash {
	hash_data(req.url);
#  	std.syslog(8+1, "MHV.REQ.URL_VCL_HASH: " + req.url);
	return(lookup);
}

sub vcl_backend_response {
    
    # Happens after we have read the response headers from the backend.
    #
    # Here you clean the response headers, removing silly Set-Cookie headers
    # and other mistakes your backend does.
 	
 	if (bereq.url ~ "^/documents/" || bereq.url ~ "\.(png|gif|jpg|swf|css|js)$") {
   		unset beresp.http.set-cookie;
   		
   		# Strip out query parameters such as 't' that do not affect the page content
   		set bereq.url = regsuball(bereq.url, "([\?|\&])+(t)=[^&\s]+", "\1");
   		std.syslog(8+1, "MHV.BEREQ.URL_VCL_RECV: " + bereq.url);
 	}
}

sub vcl_deliver {
    
    # Happens when we have all the pieces we need, and are about to send the
    # response to the client.
    #
    # You can do accounting or modifying the final object here.
    #    if (obj.hits > 0) {
    #            set resp.http.X-Cache = "HIT";
    #    } else {
    #            set resp.http.X-Cache = "MISS";
    #    }
    
    return(deliver);
}

sub vcl_backend_error { 
	return(retry); 
}

sub vcl_synth {
    set resp.http.Content-Type = "text/html; charset=utf-8";
    set resp.http.Retry-After = "5";
    set resp.status = 500;
    synthetic( {"<!DOCTYPE html>
<html>
  <head>
    <title> My HealtheVet is Temporarily Unavailable </title>
  </head>
  <body>
<div id="content-wrapper"><table cellpadding="0" cellspacing="5" border="0" style="margin:auto;">
  	<tbody><tr><td valign="top" align="left" class="welcomeBox" width="400">
    	<div class="squareCell" style="width:400px;">
<img src="data:image/jpeg;base64,/9j/4AAQSkZJRgABAQEASABIAAD/4TIaaHR0cDovL25zLmFkb2JlLmNvbS94YXAvMS4wLwA8P3hwYWNrZXQgYmVnaW49Iu+7vyIgaWQ9Ilc1TTBNcENlaGlIenJlU3pOVGN6a2M5ZCI/Pgo8eDp4bXBtZXRhIHhtbG5zOng9ImFkb2JlOm5zOm1ldGEvIiB4OnhtcHRrPSJBZG9iZSBYTVAgQ29yZSA0LjIuMi1jMDYzIDUzLjM1MTczNSwgMjAwOC8wNy8yMi0xODoxMToxMiAgICAgICAgIj4KICAgPHJkZjpSREYgeG1sbnM6cmRmPSJodHRwOi8vd3d3LnczLm9yZy8xOTk5LzAyLzIyLXJkZi1zeW50YXgtbnMjIj4KICAgICAgPHJkZjpEZXNjcmlwdGlvbiByZGY6YWJvdXQ9IiIKICAgICAgICAgICAgeG1sbnM6eG1wPSJodHRwOi8vbnMuYWRvYmUuY29tL3hhcC8xLjAvIj4KICAgICAgICAgPHhtcDpDcmVhdG9yVG9vbD5BZG9iZSBGaXJld29ya3MgQ1M0PC94bXA6Q3JlYXRvclRvb2w+CiAgICAgICAgIDx4bXA6Q3JlYXRlRGF0ZT4yMDEyLTA1LTAzVDE3OjA4OjU5WjwveG1wOkNyZWF0ZURhdGU+CiAgICAgICAgIDx4bXA6TW9kaWZ5RGF0ZT4yMDEyLTA1LTMwVDE3OjU2OjEwWjwveG1wOk1vZGlmeURhdGU+CiAgICAgIDwvcmRmOkRlc2NyaXB0aW9uPgogICAgICA8cmRmOkRlc2NyaXB0aW9uIHJkZjphYm91dD0iIgogICAgICAgICAgICB4bWxuczpkYz0iaHR0cDovL3B1cmwub3JnL2RjL2VsZW1lbnRzLzEuMS8iPgogICAgICAgICA8ZGM6Zm9ybWF0PmltYWdlL2pwZWc8L2RjOmZvcm1hdD4KICAgICAgPC9yZGY6RGVzY3JpcHRpb24+CiAgIDwvcmRmOlJERj4KPC94OnhtcG1ldGE+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgCjw/eHBhY2tldCBlbmQ9InciPz7/2wBDAAcEBAQFBAcFBQcKBwUHCgwJBwcJDA0LCwwLCw0RDQ0NDQ0NEQ0PEBEQDw0UFBYWFBQeHR0dHiIiIiIiIiIiIiL/2wBDAQgHBw0MDRgQEBgaFREVGiAgICAgICAgICAgICAhICAgICAgISEhICAgISEhISEhISEiIiIiIiIiIiIiIiIiIiL/wAARCACsAZADAREAAhEBAxEB/8QAHAABAAIDAQEBAAAAAAAAAAAAAAUGAwQHAQII/8QAUxAAAAUDAgIFCAYDCwoGAwAAAAECAwQFBhEHEiExEyJBUVIIFBgyYXGU0xUjQoGRkmJyoRYzNENTdIKisbKzFyQlNTdERWODwjY4VISjtNHS4f/EABsBAQADAQEBAQAAAAAAAAAAAAABAgMEBQYH/8QAMREBAAICAQMCBQIEBwEAAAAAAAECAxEEEiExE0EFIjJRYRTRM3GB8CNCYpGxwfHh/9oADAMBAAIRAxEAPwD9IgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAGQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAwEZcN129bkTzutzGobBnhHSH1lq8LaCypZ+xJCa0m3hE2iFfRet61zjbFvKZhH6tRrSziIUXeiMklvmX6xJGnRWPM/7Kdcz4hrVlGodOpzlSrl4U2kRmy6y0QC6Ij7jU+8Zn9wmvRPbpmUT1fdRGtV9SmlKkW9X6HecVji/CQ2qFN2lzU20Zlv/AKOfcNvQr7xNVPUn+boulWrNE1BpLsiIg4lSiKJudAcMjU2o+RpPhuSrHPAwzYZpLXHk6lyGS4AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAADw1JIsmeCLmApEu+61ckt2laettvpaUbUy4pJGcBhReslgiwcp0u5J7S7TGvpxXvb/ZnN9+EhbOmtEo8z6XnLcrNyr/AH2sT8OPe5lPqMILsS2RCLZJn8QtFFnGazHJiRJTRtSWkPNHzQ4klJ/A8gK+5pXpu5O89Xb8A5Wc9J0COffjGM+0aetf7q+nVIqtS2duEU6O3nmpttLavzI2mKdcp6YU6k3rOtvVA9Pq2+cmBUGSlW/NdP61Oc7orqvt4NCtij49hjWce6dUf1ZxfVtOijFqAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA8AatVrVJpMU5VUlMw4xfxr60tp/FWBMVmfCJmIUaq+UlpHT3DbKpqlqLgfmjLjpfmwRftG1eLeWc56w1oXlQ6SyXSbclSYxH9t+M4SfxTuEzw8iP1FVyiX9ZkuhPV6NVortHjp3yJaXU7Wy/T7Un7D4jH07b17tOuFPirrWrBecPk/StNzP6lni1MqyS+04ZYUzGPwl1ll7Br2x/m3/Cne38nQ6fToVOhMwYLKI8OOkm2WGkkhCElyJKS4EMJnbVsAADwzAc71L8oC0LKdXTm81SvJ9aEwotrZ/853iSPcWT9g6MPFtf8QxyZ4q5lF1512u+WpFpUtBNpPBlFjG8SP13nj2Z/AdH6bFX6pYxnvbwwO2n5QlbvmhXFctMdddp0lgvOEebI2Mk9vV1WjLOMn2CfUxVrMR7nRebbl+mO0ec7HoAAAAAAAAAAAACNbuajuXA7bxPF9LMsokmwfAzbWZllPfjHHuEdUb02njXjH6uvkmdbSCFpWnKTIy9nESxfQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAwzp0OBDdmTHkMRWUmt15xRJQlJczMz4BrZtxG9PKSq1WqJ27pfBXOmK6v0gbZuGftYZ7v03MF7B24+LEd79nNfPvtVCwfJx1RvCSVUvqsebOL4ml1Ry5BF3YI0tI9xGL/q6V7VhWOPa3eZWaP5IdmoaJL9WnuO+NPRIL8u1X9oz/AF9vst+kqrN4eSZW4UZUq1qiVRNBZOFJSTLivYhZdQz9+BpTnb+pS3E+zFoZoHNqs1ytXfFcjUiO5tbpjpGhUp1o+Juo/kkKL+kfsDk8r2qjBgn3fpFDaEIJCUklCSwlJcCIi5ERDz3a+gAAMByLyh9ZHrWhlbVBd23BMRufkp5xWFdpf81f2e4uPcOri8fqnc+HPyM3T293EtN7QodanOVi76iinWtEXmY+6v66U6fHoWi4rUZ81qIjwQ7c15rGqx3cuKkT3s7RH8pXRygRG6ZRIkvzBgtrSIsYkNkRd29SDP3mOKeJknvLqjkUhvwvKm0pfwTrkyKf/NjmZf8AxmsRPCyJjk1WekaxaY1ZRJh1+JvPkl1fQn7vrdgynBePZeMtZ91njSo0lsnY7qHWz5LQolF+JDOYaRL5nrkIhPLjFmQlCjaLG7KyLqljJZ4+0Fqa6o34QGnlSvedS33LtiIhzUvrJttHg5l3lgs4IyM8imOZ18zt+J4+PS8ehPVXTV1I1Mj2U1E6eK6+qYsktOJxsIkqLpSPju3Eg8pLHExGXL0L/C/hc8uZ1OumO600+WUyG3KShxtLqSUlDyTQ4RH4knxI/YNIefenTaY86VS9dWbetSox6fLNRy3HmyfRsURNx1+s+SsYUSfYM75orL0uB8Gy8ms2r4iJ/wB/stNLqUOpQW50JzpYj5b2nMGnKe/CiIxpE7ebkx2pbpt5hV9UNRGrRpDv1Mg5shpaYMhDe5hLxkZJ3r5JwfHAzy5OmHo/CPhs8rJ5jpie8b76Y9LtSHrxp6MwZDbkZtKJk1RJKOp8iLcTZ5yZnz5cBGLL1LfF/hf6W/1RO/Ee+vytNVjVCTT3WIMo4UtZYalEhLuw+/YrgY1l5uG1a23aOqPs/NV2WvqBIvSpxHDfrNahpSqTLjJPPRKQRoPCcY6p42kPOvW3V936FwuXxY49J7Y8dvET9/7936KspmAxadNagtLZipjt7WnSMnEnjrEslcd27OfaO+nh8HzrWnNabd53KWyWcZ49ws5noAAAAAAAAAAAAAAAAAAAAAAAAAAGAibsu6g2pRXqzW5BR4TJc/trV2Ntp+0pXYQtSk2nUItbTiiKVqFrxUCnVBblC07aXmKwXrv4+0RcnF/pn1U9mR17rh8d7OfVsn4h1WkUbTnTSjJZj+bUmKs9qpDyiJ15XetxXWWYwiMmWe25aTamLzqFpZdaeZQ60oltLIlIWk8kZHyMjGMw1iX2AAPMAPQAAARV3XNAtm2p1dnfweE0bhp7VK5IQXtUoyIWpTqnSLW1G35Lt60L+1XuqbPjo3vyHjdnz3spjsb+SM9u1PBKE8cD1bZK4o086KTkl1im+R/QCjJ+k65KclfaOOhttsvcSycP9o5J50+0OiOJDHUfI9pKk/6Mr8htXdIZQ6X9XoxMc+feCeJCp1nyUtRYeTp0iFUUdyVqYX+Dhbf6w2rzqsrcSfZSa5pZqJRNx1KhS0Np4G4hvpm/zN7yG0Z6T7sZw2hDQKxV6S/mnzJEB8j/AIlxbKs+4jIXmsSjdoXOh+UJqzSNqSqvnrJfYmtod/r9Vf7RjbiUn2aRyrLxRfK/nJwmuUJDne5DeNH9Rwlf3hhbgfaW1eYs7PlE6M15JIrTDrCthoPzuL0qSJXMiU30h9ncMLcK383Ri+IdP0zNVxpuselk7amNcMLcfJDjnRK/BzaYpOC/2T6tZalz23pve8lEudU2X3mmuiimxKbw0Zq39IkiM8qyRc+AxyYN+Yejwfi+TjRrHMd53P5/C3x5ERLSUE+he0iLcSk9nuE6cM23KJvG3KPdNKKl1GRsg9M288lCyI1k2edhn2EfbgRenVGnTwubbj366/VrTVtuk2TZaJbUCe1EgSnOm81dkI6NpeMK6Pee4iMiLgZhTD0+DmfEb8jU5JibRGt/u9qWrumVNMyl3BBSovspeS4r8EbjGsYLz7OP1aqbV/Kc0ppK3nKamRPlvGRrVHY6LpFEW0ty3dhngiwXAa14V5/CluXE9vOmGm39rdf3C2KM1bNGc/4vUSU65tPtabMk7j/o49omcWOnmdyiL2t47L/ZNjM2y0+89Ok1WtTdpz6nMWanHTRnalKPUbQnce1KSGF8nV/JrWulhFFgAAAAAAAAAAAAAAAAAAAAAAAABHXLclHtuiSa1WHyj0+Knc4s+fsSku1Sj4EQmtZtOoRM6cltu0K3q/XG72vdlcW02TzQaArh0iOx14u5XM/F+rz6r3jFHTX6vdjFeudz4X+8rxlW67Ao1EpLlQqk5CkwmGi2MNE3giNwy4JQWfwFePx4ybm1tRCvI5E01FY3MqtctNj2jQ37ruZ1NYvmYXm8DpC3NMuuFwbitHwJKOZnjJ/eOrDb1bdFPlxx5/8ArkzV9KnXf5skp+NcUSyLVpNCeQ5Or6YidtNYwbp7U5cWsz6rbae1SuAwtinNkm3iu/LeuWMOOK+ba8JmzLtRclrR7gWz5k0+S1GhxRGSSQo07t/VLbw54GPIwenk6PLbBn9SnX4SsCowKhH85gSG5MfJp6VpRLTkuZZTkhlasx57Nq2ifHdsCEvh51tltTrqiQ0gjUtR8CIi4mZn3EAhbd1Asy46jIp9CqjE+VFQTj6GD3ElKjwR7vVPj3GL2x2jyrFolOiiyo6padKv2isUZyouU+CiQl+STSCWp0keqjJmW3Bnn3jTFl6J2pkp1dk5bVt0W3KMxRqPHTHgRywhBczPtUo+alK5mZilrTadytWuvCSEJAEJdF9WhayCXcNTYgb0mpCXVddSU8zSgsqV9xC1cc28KzaISsSSxKjNSY6t8d5CXGll2pUWSP7yMVWR9dta0qqws63TYklpJGa1vtIVgi4me4yyQtF5jwrNIlzBjSjydLzqcmBbrpFUI6CdfKmPr2pQo9petva59w6PXy08sfSpZG1TyP6erJ0qvvN9yZLCXP2oNsxeOfP2Unhwrs/ySr8ZycOoQJJdhGbrSv2pMv2jWOfX8s54lkNJ8mXVxvlBjPJ/Qkt/92Bf9ZjR+lsj1+Tvq0gjMqFnHheZ/wD2E/q8f3R6F30z5P2sSkkaaM4jPYcltP49cP1ONPo5EhH8mfV9/wDfIzLST/lJhf2JNQrPLxrehkTNP8kK8JGDqNTgx0nzJJOvqL7jJBftFJ51fsmOLb7rXRPJDtOOZHVqrKln2tsJRGR/3q/aMrc6faNNI4se7oVr6Rac2yonKTR2EyS/3l0umd/O5uMvuHPfPe3mW1cVYWnBDNd6AAAAAAAAAAAAAAAAAAAAAAAAAAPh11tpCnHVEhtBGpSj4ERFxMzMByWHCe1kusqtNSZaaUV4002MfAqnKQeFPrLtZTyLv/EdO/Tr/qljrrn8OtoQlCSSkiJJcCIuBERdg5mzFLlRokZ2VJWTUdlJrddVwJKS4mZmERMzqEWtERuXBn7sqV+alNy4EbzrzPJUKE5waSZH/CZHclP74rv4JHvxx4wYNT239U/9Q+fnkTnz9u+vH7ykLyOJTX12bDndNXKlh677hdPrIYItykZ+wgk/YT2YLmYz4+7f4kx8sfTVryJis+nE/NP1WWxikQ5tATULjSqn2LTGSOBRl5QbjTRdV+aRYMzV9lr8eI45yTFtU75LeZ/6j93ZFN13ftjr4j/uf2eaWVSDT7Kq91ymkwKTKmSJrMdJElDbCcISSUlgsmace0xPNpM5K4472iIj+qODeK47XntWZmV3odZi1mkRarFJRRpbSXmyWW1RErxEOLJTotNZ8w7seSL16o8S9rZJVRpqVFlJx3SMj/UMUr5Wnw/Mfkz3bblqVysVGvS0QoXmKUpNXNSumIyShJZNR47h6fLpNojTi494iZ27vZ2tenV3VH6Mo8/NQMjNuO82tlThFz2byIlY7uY4cmC1PLqrlrZI3lqTZlmtNruGeiM46WWWMGt1ZFzMkIyePaK48Vr+FrZIr5a1nat2BeD5xaFUUuzSI1earSpp00lzNKVkW4vcJyYbV8orkifCRue+bSteOT9fqLMJKvUS4r6xX6rZZWf3EK0x2t4Ta8R5V2h+UDpVWqkinRar0clw9jXnDbjKVKM8ERLWRJyftGluNePZWM1Zct8sdtP0xQXPtHFlFn3KR/8AkdXA8Sw5XmErrPqTOt/T+2aNQKm5Crb0aK9JSzwc80OOZZ3YPGVkXI8ivHwxa8zMdjNk6ax90/o/qXbVVsOHbTtTXMurzGS5MZd6VxzJblLNbiixyUXaMs+KYtv2aYsm6uX+TNdtuWrW6vUq7LRCh+YNpSpfNSukztQksmo/cQ6+ZSbREQw494iZ27tZ+tunN21L6LpE8/pBWeijvtrZU5jiezeRErBdg4L4L18uquWsrlkZNFNq2tdhUyvIozspTjpr6N2Q0ncw0ozxha89/PGcDsp8Oy2p1acd/iOKtunbZnauafwrnbteRUklXHHkRyjbHODjnFBKVt2luyXHI54w21v2dPqRvSVum77etWl/SlelFEg9IlonVEo+uv1SwkjPsFaUm06ha1ohGlqxp/8AuZ/dOqqNoohuKZRJWSk73E80tpMiUs/cQt6Nt691fUjW0fbevGl9w1FFNgVQkzHT2stvoWzvUfJKTWREZ/eLX416+YRGasqR5TWp9RoyoNCtyquwqplTtRbYyhzoVo+q6+O08+qeRtw8PV3nwy5OTXaFp0T1GtmtW3SqAzUVTLjjQULnNrJxThGngs1uKLB9Y+8ZcjFNbb9mmHJuHRRg1AAAAAAAAAAAAAAAAAAAAAAAAAAwHO9Rpc+7q83prRnVNMuITJuic3/EQjPqxkn/ACsnl7EjbH8sdU/0Z379l7plMgUunsU+ntJYhRkJaYZRwSlCSwREMpna8RpsiEuL623pUa1WmrBt8lOqNxKZpN83XT4pZ/VRzX//AAez8N48Ur6t/wC/y8T4lyZvb0qf1SbsalaO2Ct1rY9dNR6nTeJ7HZn+KZ5+0/eMom3Mzf6Ia6rw8P8ArlE6J6dyao+q77h3OsvL6SK07/vDmc9O5nmkleqXafHsIbfEuXFY9OjH4bw5tPqXZNW7gnXfeETT2ir+oJ5JT3E8jdLief0WU5M/0vcI4GKMWOc1v6f3+VudlnNkjDX+r2rKg3VV0WxDd8006tRKTqsrOEOra4bN3bywX3n3CKbxV6575snhN9ZJ6PGLH5T1Iq1w13USBEi7qdbtLi+cqpqOqom3E9HGKT3KcLrpb+ynnxHPkpSmGZ83tPn/AJ06Md73yxEdqVjx+6+Vn/U8z+bu/wBwx59fL0Z8Pyx5O+n1t3pdcuNX21vw4Mbp0R0qNCVqNwk4WaetjB8iMeryss0r2cGDHFp7t2bQKVbflMQKVRmvNaezUYZtNEoz2E4gjWRGrJ4PJivVNsO5TrpydkFOuqgVXWKZXr7J2VREzHycjt9Y+jYNTcdoiM09TKSzxGkUmMeq+VZtu/divG7rJa1BgXJp405TY0dTDpsqIm9r6F9fYklK6q0YIy94Upbo1fuWmOrdXYddKNo2uv0y4bylvtTTY2qpsLrPSmuaDXjihKDM+OSzyHHxrZO8VdGbo82cV1NqmmVQkxnbCgP05htCky0PGeFKI+otOVuGR888R24ovH1Tty5Jrvsu3lPuuO06zXXFGpxdKWpaj5mZpZMzMYcPzZtn/wAqS8oikUxOm9oVhLCSqa240Zcn7ZslFNZI927iI4k/PaDkR8kLvolaFsw9LYdwxYLTdbmU51MqaRfWLLcvgZ/0SGHIvPXr2bYaxFXHfJ10/ty9LnkR6+2t6HBiJkIjpUaErUayThZpwrGO4x2cvLNI7ObBji1p23Ljt6lWn5SFNplCb82gtT4C2WSMz2dMlJrIjPJ4yZ8zFa2m2Gdp105OzrOqd51qp1pvT60lf6Sk/wCsZSTx0TeMmjcXq9Xis+7hzMOFx61r6uTxHhjzeRa1vSx+fdAQ7Uoyqc7aVkRE1WqSi6CsXQ+jMZhJ+uTKjyRq7iR+JmN7Z7b9TJOojxX3YVwV16eOOqZ82VXyl7fj0qvUuv0h8nJMZtmDNWk8qRKipJxhTmOSlt/2DDjWmd79+7syxEa17dnvlDX6u7aZaNOp5b/PoyKktpPa/I+paRj2K3ivFx9PVP2Wz33qFV1WbZpl2060Za1fQtuRosVaW+9xKXpjqS4ddxSz4jXB3rNveVMs99ezzVa4NIKrGp52HBdpsuPuRJ3JJslowWxXrrysldvMMFbx9fgyzXt0wnNZ3YdX03sq7VtpVXKjHU1Oml67vm7ZJLcfsVkxnxu17R7LZvpifd27Riz7YpNm0urU2A1GqVQgRzmSUF13D2krrGeftHkcee8zaYdOGsRVeBi1AAAAAAAAAAAAAAAAAAAAAAAAAEJfN2R7XtqRVltm/II0swoifXkSnT2MMp9q1mX3C2OvVOlbzqGpp1aL9u0RaqispFw1J05taleOS5zSn9Bsuogu4hOW/VP4KV0swosj7kqyKPQZ1VX6sOO4/wDkSZkLYqdVor95Uy36azP2hQ9ErDdhRFXhWUmqu1Ujcb3+s006e4z4/ac5n7B3/EuV1T6dfpq4PhvF1HqW+qyNum15t+6xrpsrcmg0Nlnzk+XBwukNCf0nT4e4htgzRg4+4+uzHPgnPyNT9NXS69NRQ7XmTI7ZEiBFWtlpJcC6JHVSRdxYHm4q9d4ifeXp5bdGOZj2hxnTqiVaHYVw34lKnaxIYebgKwZr6x/XvFjjncfDHcPY5mWs5aYvFY8vG4eG0Y7Zf80+EjRo9Mt21YlQrbSo1Hi/W0qju8JNTqB8pL7fPaSsdGjsLifYM8kzkvMV7zPmfasfaGmOIpTdu0e0e9p+8rTp/ZtxnBVNuRXm0me+c6bGZUfSvuq9QpDhckNpIiS0ngXbkcnLz03qnt2/v93XxMF9bv79/wD39lyrPCjTM/yDv9wxwx5d0vzt5H60HeNYIjI/8wL/AByHoc6flhycaPmljvN1tPlXRcmWSqFP4f8ATSJp/ARf+Ijrpo8jSvWNdWq1LKo23IkPPsE6hK2nWZOTUgjWRoJ1o1cCPuFqW9THqJ7ovHRffst7euOmtUr1NpduWSiYcl9CJOYcfpUtqPBm222S9xp58TIhlPHvEbtZrGasz2hUvKIiSKbrJ9IVaOp2kvFEcYSZYQ7HZJJOspVyzwURl7RrxJ3j1HllyI+bfsjNbL4tO8ZcF206eqJS6bHNt9fQoZQS3VZSRkjgWMY48+wW42OaeZ8q5rRaeyd1/kv16y7KuRmM6zT3ILkdSnEGjY7hvBH3ErYZpPtIU43a1oaZ/ES3tUbiiXpoVQqnRm3Fs0SU1FqadvFlSI3R7lYz1MqLrCuGOjJO/cy/NTssWh+rtszLRp9goQ8mtsQZRKUZF0Jk0Sl5JWc8SPuGfJwz1dXs0w5Pl0q/kfOIVd9WIjyf0c3/AIpDXn/TCnFjvL51NcaLypqeSlF/DaXw/ooEYv4EotH+IuumNate359y1i7JzMWvOzXWXmnz+uJslbj2o5mS1H2dw6Objvkilccbrpw8K9aTe2SdW2ssauXRd6Ch2pGXQbZLqrq7zZNvLR3Q2Ps58ahyzipi75J6r/b93XGS+XtT5a/f9nzqVplTpektQt+kMYdjoObFUZ7nHJTX1hrWs+KlucSMz7xjTkT6nVLonBWKahwvydbccujUqA/Iy7BobXnSiPiSTb4MN/c4rP3Ds5d+mn83Px6TNu6x+UlZtWol+M33Fi+dUmQbC5Bmne2iRHwWx4uOEOISXE+HMZ8TJuvT7r8inzdTcneUJpYqlp+iLMZkVxwiJMZyLG6LefMiUglrX7MJ4+wV/S333t2W9ev2bHlMwly9PLZqkKmLp9NbUo34vRdH5r5w0RpS4hPBHWIy94cOfnncnIiZrCzaC6w21XqXSrNaS8mtwoBdMakl0R9BhB7VEZ5yR55DLk4JrPV7NMOTcadXIczZ6AAAAAAAAAAAAAAAAAAAAAAAAAj6lbtIqU+BOnM9M/THFPwtxntQ6pJo37eRqJJngz5CYnSJhvkQhL0B8PsMyGlMvoJxpZbVoUWUmR9hkfMIJjb0kERbS4EXIgHymOylanEpInHMb1EXFWOBZPtwQI0SY7MmM5HfSS2XUmhxB8jSosGX4CYnRaNxpVplRjWvCiWpbDCHJzTOWkPLPoIkZPORKc5kkuwuaj/EdFazkmb3/wDfxDlvf046Kef+PzKt2ZbTV0XQV0zVqm02As/M50gsLnSi4G8hHJqM1yaQXM+J5MdPIzelTojtM+32j9/u5sGD1L9c94j3+8/s6hgea9QNJHz5dwDCxAgx1GphhtpR8DNCUpP9gbNBwIKnunUw2b3PpDQndw9uMhs09lwYUxg48xlD7CvWbdSS0n9yskETo016bb9DpZGVMgx4e7n0DSG8/lIhM2mUREQ+6nRaRVY3mtTiMzI38k+2lxOfcojCLTHgmIlrNWharNMdpTVLiIpj379ESw2TS/1kYwYnrt9zphuv0+C/F80fYbciYJPQLQlSMFyLafDgK7Tp8RKRSocZUaHFZYjL9ZpptKEH2cUkREYmZlGnxCoFCguKchQY8dxXrKaaQgzz7UkQTaZIrDMxAgx1GqOw20o+BmhCUnj7iEbNC4EFb3TLYbU9z6Q0JNXDlxxkNp0wPW/Q35hTn4Md2anlIW0hTnD9IyyLRktEa3OlJxVmdzEbbu0VXMcMAMTEGHHMzYZQ0avW2JJOffgNmn2tltxBtuJJbauCkqLJGXtIBowrXtqBIOTBpsWPIPj0rTLaFfiREYmbSjphvOsMvNqadSS21lhaFERkZe0jEJa0ChUWnqNUCExFUr1lMtIbM/ykQmbTKIrDcEJAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABQpemVbqtfqSqvU0nbU6QmQ5BjpND75ISSW2H3ufRIx6qeY7a8utax0x88Q4Z4lptO5+WZXiLGjxY7ceO2lphpJIbbSWEpSXAiIvYOKZ35dta6jTKCQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAfmD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAPSb1T8UD4ZXzQD0m9U/FA+GV80A9JvVPxQPhlfNAc1AAAAAAAAASkC0rmqFGkVuDAdfpETd5zLTjYjYWVZ4kfAjAYKJQazXZ6adR4q5k5STUlhvG4yTxM+OC4AMM+BMp816DNbNmZGWbT7KuaFp5pP3AMAAAAACTYtS5JFAduFmC4uhsHtemljo0mRkRkfHPNRdgCMAAABvUK3q3X530fRYi5k3abnQt4ztTzPiZd4DVlRpEWU7EkoNuSws2nmz5pWk8KSftIwGMBJVO1LkplLi1WoQXI9Nm480krxtcyW4tuDzy4gI0AASVOtW46lSpVXgQXH6XCz53JTja3gtx7snnlx4EA1abTZ9TnMwKeyqRNkHtYZR6y1dxZwAkLgse77cZbfrtMegsvK2NLd24UoizjgZ9gDPRdOL9rcBFRpFHkS4DhmSH0bdp7TweMmXIwEE42ttxTay2rQZpUnuMuBkAy0+nzajOZgQWlPzZCibYZT6y1nyIsgMtaodXodQXTavGXEntkk1sOY3ESiyk+GeZANMBkajyXUOraaW42wRKfWlJmltJntI1nyTkzxxAYwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAdm0u/wDLxd/vk/4KAFe8m3/atE/m0j+4AnomlVKvi87+mTZzkJ2m1B3olJ29H1t6tzmSM9pbOOOwBpQ9JtOrismqVWzq3KmVOioWqSbzZNtOmhBuYSjalSUrJJ7TyYD1Okdi25aFMrt7y6mpdVSlZFTGekajktJLLpVEhZ8CUWTP8AEHU9LKbMv2Bblm1pirQ6iRrKQSiUqMlBbnOm2cDwniWMZ5ALGelWjL1Yl2nHuWS1c8JCjdffJCYvSI9ZPFJJ6ueJEvICbsKhwV6AV2kVieiHCZnyUzJ7f1yCQy62alN49fdt6vvAVG5tMLIkaauXzY9RlTIsRfRymJaSJZnvJCiItqDSojUR4AbUjSzTG0oVNY1Frr8SvVVBLSxFIujYJRkWVmaFngjPBqPhkBha8nypq1MK1TmZpHm30gVSJPE4u7YREnl0m/h3doC46PW3pfG1Dfds2syJcuAw6zLiykcHEqUSelYcJKMklRYMBS3dOKZW6Jft3PyXm5lHqM4mWEbejXsInC35Iz4mrsARll6b0uv6c3DdMiS81LpBLNllvb0a9rW/rZIz5n3gL1elt1C6NNdOaFAwUqb0SSWr1UJKLuWs+3CUlkBFJ0n0fl3E9Y8C5JP7tGkqLK0o83U8hO5TeCSXFJcTIlZARlm6MQvMa3Wr8mqplFob7kV/oMGtxxnG5SVGSur1iJOCyYC52xSbThaK3dJtSouVCkTEPLT06Nj7K0tElTbhYLjyMjwA5Vo7/tPt7+dp/umA7Trqabh05uNtKd0y25bbxEXPaSEr/a26YCy6doRb9Kt+zyIunTSzmSPYrcglfit1X4AOFUa0dMkUWXct7V1TBOTHWmKXBNKpJfWKIlLRha+ODPgWMdoCXrGklMoV02fUbaq0n6Kr8ptMaUZJTJYNSd6FoPaRHlJ/aTwASTmjse59VK/RqvWJsl6BEivNznOj6VZuo4JcPbjanswQCPt3R7Ta6KZU4Fu1+RLueloM3lm2SIql8cbEqSRm2ai27twCy21TdOEeT3LS9UX2qE64lVWnpa+ubkJdbNbZFsypKHMJLgfABTkaYadW1bUSu6hVmRERVlqOmRYpF0nQH1kLcylZ56MyUrsLOAHsjQ+CxqFQ6Q3UVSrXuFp16DUWtvS4ab6TafA08jIyMi4kAkoOjelX7tJFlS6/LcuE9yorTKEkhtJJ3khxZoNKndvWMixwAV2i6aWrFr9wxbzrzdPp1vuG2ZNqSmTJ6pLJTaFbuG0y5EZ5AbF5aa2V/k1O/7JqEqRTmXCbfYmp2q/fCbVt6qDI0qMvYYCUubR/TK0mKTUrkr0tinVBBZYQ2S31umlKj27UGSG0EeVGZGA0bz0Jcg3vRKHbks5MGvJW4y+/6zKWiJTilGnBKLYeS4F3AJA9KtGXqzLtKPcslu54SFG7IfJCYvSILrI4pSnq54kS8gIa1tKKC1b1Quy+6p5jb0GQuI0qGZOKkLbX0ZqbXhRGlSuCcFxAa9+abUCHacS9rMqK6lbEhzoXumIieZWZ4LOCTw3FtMjLJAKEAAAAAAAAAAAAAAAAAAOyaCyqbXLEuWwXJSItTnk4qJv+0l1okbi8W1RcSLsASWkOkFzWPdLtz3W9Eh02FHdSS0vbiUayxuMzJO1JFx48QH3pbWI9Zgan1eP/B5y3nmc8MoUy9t/YAh/JyMism9P5qXD/27oCes17V+i2XSHrSkwb0orrZJ81cxHdjJwWG+lNfWIuWFFkgEjXqvY1saoWvUJTMamV6aw8xV0RzTsa6ZCejN5SSSX76RpJRlyAVOq+TzXqrf9XnVGW1DtOQ4/NRVSWhR4dPelOw8YwZnuM+GAGalR4MbyarmiwJHnUJqe+iPJ27OlbS+0RL29mQGnZL/AJv5MdwyMbjanrd29qujeZVj9gC+XlJvS6foit2BFo1VpExlKZDk5pDrrCjVnJmpRdUiPikuJGXtAa1v3hMjapvWvdVXp0uU/TjYiSILJx0tuGvPm7m5bhbzT1iIj/aAiNF9J7is7USTMrrjDLXQvMwEpdSpcojUSjcSjmSEpLjntMAsKH9P2zqXasJ1BViXUZptNLPHB1JIQfu3IMsgPLTs+p2rojd8GrraTVXWXnH4bbiXDZLocJJZp4blEWcdwCWh3DTqBbmmU+orJuGaCjuOnybN+HtSo+4t2MgM8yk6ufu1mSIDdvxLfyuTErzkVK3CQpOSJeHELNfYpXLHEBC02W9qVpxdFqxp0ORdDc1x4lskbDEkicStDjaVGpRIWaTTnjxAeWlZ1RtPRO7YFWcaKqvIeefiNOJcNgjZIkJWaeG5RFn3YAct0d/2nW9/O0/3TAdtt2bEka2XnbM4krhVKLFf6NXJRtNk2svyrIBnt2tIqHlCVuMhX+b0yjsxG+4j6VLi8fevH3AIO37ZkRtOI1Y06pVNqd1ypLn0hLnbFONn0i9+N2MGhWC25LhxASt7efSX9N5Ul9qY8zVmkTpcbHQG/wBEaV7dvAi3pURANmjGX+W28zz/AMLhYP8A6YCgeS1/4kuPP/o0/wCKoBvWJRKjc/k6VigUUkP1dc93DBrJHEpDbvWM+WUkAs1anXZcdm0aZYsSlVKVEScOqQp7aHFsOtkTa0pNak7dq0GRl2lgwEcmXdLOrdn0Ov1GmS3IpSn0xKawpg4uYpo2uZWtO08mSSLHLICs0gy9Ktw+zz+R/wDUUAt9OtulzK7flYplPhVa+o880wo0/BoQnzdo28EfIlZUe7tPhkBr6gfusleT5W2LjVFXcDKkrlRIO3ZGaJ5C0NmSM9ZLfEwFa8p48060u7zN7/DaAXe47kptv3lp/NqKyahuw34q3j9Vs3WWiQaj7C3ERAKbV/J5rtW1Bq86pymodpyHHpqKqlxCzw51kp2KxjB+sZ8MAJ6x6guo6TP23Yr0GqVqhSnG2Wqi0k0SWCeNSXSbMy29IhXVPvAQGqkjUKlacop9yyqNEKprTuoUKKbb6TSvfuS4hxSOG0jUe32ZAccAAAAAAAAAAAAAAAAAAHpGZGSkmZKTxSouBkfsMuQDPJqVRlIJEuU++2nkh11a0/gozIBgJbqc7VqLPMiUZEfvxjIAlbickhSkkfMkqMs+/BgMsWdOh58zkPR93rdC4tvPv2mQDEs1LUpSzNalcVKUe4z95nnIDKqdPXGKIuS8qIXJg3Fm3+TO0Bj3ubdm9RIPmglGSfy8gAluEnaSlEg+aCUe0/eXIBkiz58QlFFkOsEv1yacW2SveSTIBh9vbnOe3PfnnkBlemzX3CdekPOPEW0nFuLNRJ7iVnOAHjEmVHeJ+O84y+XJ1takL4/pEZGA86Z7KjNxZms8rytXWM+1XHj94D5Na1J2qUpSC5JUozIvcR8AGU58843mhyXjiF/EdKvo/wAmdv7AHwy++w6T0dxbLyfVcaUaFF7jTgwA3n1GozdcM1nlfXV1j71ceP3gPklKSeUmZGXIyPB/iQD3pHd+/evf49yt35s5ACddJW5K1ks+aiUolH7zI8mA+2ZkxhC0R33WUOfviW3FoJf6xJMsgPhLjqUklK1EguSSUoiL3FnBAHSv53E4vcfNW9WT955yYAlxxB5QtSDPmaVGn+zAD7jTJsU1HFfdYNZYX0S1I3F3K2mWQCNMmxVqXFkOsLV6ymnFNmfv2mWQHx0jm83NyulPibm49+T7d2cgG9zdu3q3+Pce783MB9sy5jD3TMPutP8AI3ULUlf3qI8mA+Old63XX1+K+srrGfM1cetn2gPFLcV66lLxy3KM8e7OQBS3F+utSscCJSjVj3ZyAyrnTlxiiLkvKiFyYNxZt/kM8APhl99hwnWHFtPJ9VxtRoUXuUnBgD778h03pDi3nj5uOKNavxVkwHwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAD/2Q==" width="400" height="172" alt="My HealtheVet Unscheduled Maintenance" border="0">
<h4 align="center">My HealtheVet Unscheduled Maintenance</h4>
		<p style="font-size:12px;">The My HealtheVet system is experiencing an unexpected problem.  The site will be offline in order to address the issue. The website will be restored as soon as possible.</p>
		<p style="font-size:12px;">If you need help with a VA prescription refill or need to confirm your VA appointment, please call your local VA medical center. A list of VA medical centers and clinic phone number contacts is available online through the <a href="http://www2.va.gov/directory/guide/division_flsh.asp?dnum=1" title="VA Facility Locator"> VA Facility Locator</a>. We apologize for the inconvenience and appreciate your patience.</p>
</div></td></tr></tbody></table></div>
  </body>
</html>"} );
    return (deliver);
}


#sub vcl_fetch {
 # do not cache anything that's not 200.
  #  if (beresp.status != 200) {
  #      set beresp.ttl = 0 s;
  #  }
#}
